﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using SixLabors.ImageSharp;
using SixLabors.ImageSharp.Formats.Bmp;
using Xunit;
using YusakuClassLibrary;
using static YusakuClassLibrary.IChainEffect;

namespace RunRunGridLinerCore.ViewModel.Tests
{

    /// <summary>
    /// メイン画面ビューモデルテスト
    /// </summary>
    public class MainViewModelTests
    {

        #region テスト用データ

        /// <summary>
        /// テスト用チェーンエフェクト
        /// </summary>
        public class TestChainEffect : IChainEffect
        {

            /// <summary>
            /// 発動結果
            /// </summary>
            private ActivateResult m_result;

            /// <summary>
            /// 値
            /// </summary>
            private object m_value;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="result">発動結果</param>
            /// <param name="value">値</param>
            public TestChainEffect(ActivateResult result, object value)
            {
                m_result = result;
                m_value = value;
            }

            /// <summary>
            /// 効果を発動します。
            /// </summary>
            /// <returns>発動結果</returns>
            public ActivateResult Activate() => m_result;

            /// <summary>
            /// 値を取得します。
            /// </summary>
            /// <returns>値</returns>
            public object GetValue() => m_value;

        }

        public static IEnumerable<object[]> ReadImageFileEffect_Success_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage001.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage002.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage003.png") };
        }

        public static IEnumerable<object[]> ReadImageFileEffect_Fail_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Fail, "TestData/Input/ReadImage004.png") };
        }

        public static IEnumerable<object[]> SaveImageFileEffect_Success_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage001.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/WriteImage001.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage002.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/WriteImage002.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage003.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/WriteImage003.png") };
        }

        public static IEnumerable<object[]> SaveImageFileEffect_Fail_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage004.png"), new TestChainEffect(ActivateResult.Fail, "TestData/Output/WriteImage004.png") };
        }

        public static IEnumerable<object[]> DrawGridEffect_Success_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage001.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/DrawImage001.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage002.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/DrawImage002.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage003.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/DrawImage003.png") };
        }

        public static IEnumerable<object[]> ClearGridEffect_Success_TestData()
        {
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage001.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/ClearImage001.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage002.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/ClearImage002.png") };
            yield return new object[] { new TestChainEffect(ActivateResult.Success, "TestData/Input/ReadImage003.png"), new TestChainEffect(ActivateResult.Success, "TestData/Output/ClearImage003.png") };
        }

        #endregion

        #region テスト用メソッド

        /// <summary>
        /// 指定したパスの画像を読み込み、バイト配列として返します。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>バイト配列</returns>
        private static IEnumerable<byte> LoadImage(string path)
        {
            using Image image = Image.Load(path);
            using MemoryStream stream = new();
            image.Save(stream, BmpFormat.Instance);
            stream.Flush();
            return stream.ToArray();
        }

        #endregion

        #region テスト

        /// <summary>
        /// 初期化チェック
        /// </summary>
        [Fact]
        public void MainViewModel_CheckInit()
        {

            using MainViewModel mainVM = new();
            Assert.Null(mainVM.DrawImage.Value);
            Assert.Equal(100, mainVM.GridWidth.Value);
            Assert.Equal(100, mainVM.GridHeight.Value);
            Assert.Equal(0, mainVM.GridX.Value);
            Assert.Equal(0, mainVM.GridY.Value);
            Assert.Equal(10, mainVM.LineWidth.Value);
            Assert.Equal(Color.Black.ToHex(), mainVM.LineColor.Value);
            Assert.Equal(255, mainVM.LineAlpha.Value);

        }

        /// <summary>
        /// 実装済みチェック
        /// </summary>
        [Fact]
        public void IntoTheVRAINS_CheckImplemented()
        {

            using MainViewModel mainVM = new();
            Assert.True(mainVM.IntoTheVRAINS());

        }

        /// <summary>
        /// 画像ファイル読み込み/成功
        /// </summary>
        [Theory]
        [MemberData(nameof(ReadImageFileEffect_Success_TestData))]
        public void ReadImageFileEffect_Success(IChainEffect readChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);

            IEnumerable<byte> imageArray = LoadImage((string)readChainEffect.GetValue()).ToArray();
            Assert.True(mainVM.DrawImage.Value.SequenceEqual(imageArray));

        }

        /// <summary>
        /// 画像ファイル読み込み/失敗
        /// </summary>
        [Theory]
        [MemberData(nameof(ReadImageFileEffect_Fail_TestData))]
        public void ReadImageFileEffect_Fail(IChainEffect readChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);

            Assert.Null(mainVM.DrawImage.Value);

        }

        /// <summary>
        /// 画像ファイル保存/成功
        /// </summary>
        [Theory]
        [MemberData(nameof(SaveImageFileEffect_Success_TestData))]
        public void SaveImageFileEffect_Success(IChainEffect readChainEffect, IChainEffect saveChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);
            mainVM.SaveImageFileEffect(saveChainEffect);

            IEnumerable<byte> imageArray = LoadImage((string)saveChainEffect.GetValue()).ToArray();
            Assert.True(mainVM.DrawImage.Value.SequenceEqual(imageArray));

        }

        /// <summary>
        /// 画像ファイル保存/失敗
        /// </summary>
        [Theory]
        [MemberData(nameof(SaveImageFileEffect_Fail_TestData))]
        public void SaveImageFileEffect_Fail(IChainEffect readChainEffect, IChainEffect saveChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);
            mainVM.SaveImageFileEffect(saveChainEffect);

            Assert.False(File.Exists((string)saveChainEffect.GetValue()));

        }

        /// <summary>
        /// グリッド描画/成功
        /// </summary>
        [Theory]
        [MemberData(nameof(DrawGridEffect_Success_TestData))]
        public void DrawGridEffect_Success(IChainEffect readChainEffect, IChainEffect saveChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);
            mainVM.DrawGridEffect();
            mainVM.SaveImageFileEffect(saveChainEffect);

            IEnumerable<byte> imageArray = LoadImage((string)saveChainEffect.GetValue()).ToArray();
            Assert.True(mainVM.DrawImage.Value.SequenceEqual(imageArray));

        }

        /// <summary>
        /// グリッドクリア/成功
        /// </summary>
        [Theory]
        [MemberData(nameof(ClearGridEffect_Success_TestData))]
        public void ClearGridEffect_Success(IChainEffect readChainEffect, IChainEffect saveChainEffect)
        {

            using MainViewModel mainVM = new();
            mainVM.ReadImageFileEffect(readChainEffect);
            mainVM.DrawGridEffect();
            mainVM.ClearGridEffect();
            mainVM.SaveImageFileEffect(saveChainEffect);

            IEnumerable<byte> imageArray = LoadImage((string)readChainEffect.GetValue()).ToArray();
            Assert.True(mainVM.DrawImage.Value.SequenceEqual(imageArray));

        }

        #endregion

    }

}
