﻿using System;
using System.IO;
using SixLabors.ImageSharp;
using SixLabors.ImageSharp.Drawing.Processing;
using SixLabors.ImageSharp.Formats.Bmp;
using SixLabors.ImageSharp.Processing;
using YusakuClassLibrary;
using static YusakuClassLibrary.IChainEffect;

namespace RunRunGridLinerCore.ViewModel
{

    /// <summary>
    /// メイン画面ビューモデル
    /// </summary>
    public class MainViewModel : IPlayMaker, IDisposable
    {

        #region プロパティ

        /// <summary>
        /// 描画イメージのカードステータスを取得します。
        /// </summary>
        public CardStatus<byte[]> DrawImage { get; } = new();

        /// <summary>
        /// グリッドの幅のカードステータスを取得します。
        /// </summary>
        public CardStatus<int> GridWidth { get; } = new();

        /// <summary>
        /// グリッドの高さのカードステータスを取得します。
        /// </summary>
        public CardStatus<int> GridHeight { get; } = new();

        /// <summary>
        /// グリッドのX座標のカードステータスを取得します。
        /// </summary>
        public CardStatus<int> GridX { get; } = new();

        /// <summary>
        /// グリッドのY座標のカードステータスを取得します。
        /// </summary>
        public CardStatus<int> GridY { get; } = new();

        /// <summary>
        /// 線の太さのカードステータスを取得します。
        /// </summary>
        public CardStatus<decimal> LineWidth { get; } = new();

        /// <summary>
        /// 線の色のカードステータスを取得します。
        /// </summary>
        public CardStatus<string> LineColor { get; } = new();

        /// <summary>
        /// 線の透明度のカードステータスを取得します。
        /// </summary>
        public CardStatus<int> LineAlpha { get; } = new();

        #endregion

        #region 変数定義

        /// <summary>
        /// 読み込みイメージ
        /// </summary>
        private Image? m_readImage;

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MainViewModel()
        {
            DrawImage.Value = null;
            GridWidth.Value = 100;
            GridHeight.Value = 100;
            GridX.Value = 0;
            GridY.Value = 0;
            LineWidth.Value = 10;
            LineColor.Value = Color.Black.ToHex();
            LineAlpha.Value = 255;
            m_readImage = null;
        }

        #endregion

        #region デストラクタ

        /// <summary>
        /// デストラクタ
        /// </summary>
        ~MainViewModel()
        {
            Dispose(false);
        }

        #endregion

        #region ディスポーズ

        /// <summary>
        /// ディスポーズ
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ディスポーズ
        /// </summary>
        /// <param name="disposing">マネージリソースを破棄する</param>
        protected virtual void Dispose(bool disposing)
        {

            /* マネージリソースを破棄 */
            if (disposing)
            {
                m_readImage?.Dispose();
            }

        }

        #endregion

        #region プレイメイカー実装

        /// <summary>
        /// VR空間にダイブ！
        /// </summary>
        /// <returns>成否</returns>
        public bool IntoTheVRAINS()
        {
            return true;
        }

        #endregion

        #region 画像ファイル読み込み

        /// <summary>
        /// ファイル選択ダイアログを表示し、選択した画像ファイルを読み込みます。
        /// </summary>
        /// <param name="chainEffect">ファイル選択ダイアログ表示チェーンエフェクト</param>
        public void ReadImageFileEffect(IChainEffect chainEffect)
        {

            /* ファイルが選択されていない場合、終了 */
            if (chainEffect.Activate() != ActivateResult.Success)
            {
                return;
            }

            /* 画像ファイル読み込み */
            m_readImage = Image.Load((string)chainEffect.GetValue());

            /* バイト配列として取得 */
            using MemoryStream stream = new();
            m_readImage.Save(stream, BmpFormat.Instance);
            stream.Flush();
            DrawImage.Value = stream.ToArray();

        }

        #endregion

        #region 画像ファイル保存

        /// <summary>
        /// ファイル保存ダイアログを表示し、画像ファイルを保存します。
        /// </summary>
        /// <param name="chainEffect">ファイル保存ダイアログ表示チェーンエフェクト</param>
        public void SaveImageFileEffect(IChainEffect chainEffect)
        {

            /* 必須データがない場合、終了 */
            if (DrawImage.Value == null)
            {
                return;
            }

            /* ファイルが選択されていない場合、終了 */
            if (chainEffect.Activate() != ActivateResult.Success)
            {
                return;
            }

            /* 保存先ファイル名を取得 */
            string fileName = (string)chainEffect.GetValue();

            /* バイト配列から生成 */
            using MemoryStream stream = new(DrawImage.Value);
            Image image = Image.Load(stream);

            /* 画像ファイル保存 */
            image.SaveAsPng(fileName);

        }

        #endregion

        #region グリッド描画

        /// <summary>
        /// グリッドを描画します。
        /// </summary>
        public void DrawGridEffect()
        {

            /* 必須データがない場合、終了 */
            if ((LineColor.Value == null) || (m_readImage == null))
            {
                return;
            }

            /* 描画用イメージを生成 */
            using Image drawImage = m_readImage.Clone(x => x.Resize(m_readImage.Size()));

            /* 描画領域の幅と高さを取得 */
            float width = drawImage.Width;
            float height = drawImage.Height;

            /* グリッド用ペンを生成 */
            Color color = Color.ParseHex($"{LineColor.Value.Substring(0, 6)}{LineAlpha.Value:X02}");
            Pen pen = new(color, (float)LineWidth.Value);

            /* 縦方向の線を描画 */
            for (int i = 0; i < width; i += GridWidth.Value)
            {
                drawImage.Mutate(x => x.DrawLines(pen, new PointF(i + GridX.Value, 0), new PointF(i + GridX.Value, height)));
            }

            /* 横方向の線を描画 */
            for (int i = 0; i < height; i += GridHeight.Value)
            {
                drawImage.Mutate(x => x.DrawLines(pen, new PointF(0, i + GridY.Value), new PointF(width, i + GridY.Value)));
            }

            /* バイト配列として取得 */
            using MemoryStream saveStream = new();
            drawImage.Save(saveStream, BmpFormat.Instance);
            saveStream.Flush();
            DrawImage.Value = saveStream.ToArray();

        }

        #endregion

        #region グリッドクリア

        /// <summary>
        /// グリッドをクリアします。
        /// </summary>
        public void ClearGridEffect()
        {

            /* 必須データがない場合、終了 */
            if (m_readImage == null)
            {
                return;
            }

            /* バイト配列として取得 */
            using MemoryStream saveStream = new();
            m_readImage.Save(saveStream, BmpFormat.Instance);
            saveStream.Flush();
            DrawImage.Value = saveStream.ToArray();

        }

        #endregion

    }

}
