﻿using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using Microsoft.CSharp.RuntimeBinder;
using RunRunGridLiner.View.ChainEffect;
using RunRunGridLinerCore.ViewModel;
using YusakuClassLibrary;

namespace RunRunGridLiner.View
{

    /// <summary>
    /// メイン画面ビュー
    /// </summary>
    public partial class MainView : Form
    {

        #region 変数定義

        /// <summary>
        /// メイン画面ビューモデル
        /// </summary>
        private IPlayMaker m_mainVM;

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MainView()
        {

            /* コントロール初期化 */
            InitializeComponent();

            /* ビューモデルを生成 */
            m_mainVM = new MainViewModel();

            /* 未実装の場合、中断 */
            if (!m_mainVM.IntoTheVRAINS())
            {
                return;
            }

            /* コントロールのプロパティをバインド */
            BindingCntrolProperty();

        }

        #endregion

        #region ディスポーズ

        /// <summary>
        /// ディスポーズ
        /// </summary>
        /// <param name="disposing">マネージリソースを破棄する</param>
        private void DisposeSub(bool disposing)
        {

            /* 未実装の場合、中断 */
            if (!m_mainVM.IntoTheVRAINS())
            {
                return;
            }

            /* マネージリソースを破棄 */
            if (disposing)
            {
                ((dynamic)m_mainVM).Dispose();
            }

        }

        #endregion

        #region フォームイベント

        /// <summary>
        /// フォームロード時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void MainView_Load(object sender, EventArgs e)
        {
        }

        #endregion

        #region コントロールイベント

        /// <summary>
        /// 終了メニューアイテムクリック時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void MenuItemExit_Click(object sender, EventArgs e)
        {
            Close();
        }

        /// <summary>
        /// 画像ファイル読み込みメニューアイテムクリック時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void MenuItemReadImageFile_Click(object sender, EventArgs e)
        {
            try
            {
                ((dynamic)m_mainVM).ReadImageFileEffect(new OpenFileChainEffect(this));
            }
            catch (RuntimeBinderException ex)
            {
                Console.WriteLine(ex.ToString());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// 画像ファイル保存メニューアイテムクリック時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void MenuItemSaveImageFile_Click(object sender, EventArgs e)
        {
            try
            {
                ((dynamic)m_mainVM).SaveImageFileEffect(new SaveFileChainEffect(this));
            }
            catch (RuntimeBinderException ex)
            {
                Console.WriteLine(ex.ToString());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// グリッド描画ボタンクリック時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void ButtonPaint_Click(object sender, EventArgs e)
        {
            try
            {
                ((dynamic)m_mainVM).DrawGridEffect();
            }
            catch (RuntimeBinderException ex)
            {
                Console.WriteLine(ex.ToString());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// クリアボタンクリック時イベント
        /// </summary>
        /// <param name="sender">イベント送信元</param>
        /// <param name="e">イベントパラメータ</param>
        private void ButtonClear_Click(object sender, EventArgs e)
        {
            try
            {
                ((dynamic)m_mainVM).ClearGridEffect();
            }
            catch (RuntimeBinderException ex)
            {
                Console.WriteLine(ex.ToString());
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        #endregion

        #region コントロールプロパティバインド

        /// <summary>
        /// コントロールのプロパティをバインドします。
        /// </summary>
        private void BindingCntrolProperty()
        {

            /* グリッドの幅 */
            SetBindingProperty(PanelParam.GridWidth, ((dynamic)m_mainVM).GridWidth, "Value");

            /* グリッドの高さ */
            SetBindingProperty(PanelParam.GridHeight, ((dynamic)m_mainVM).GridHeight, "Value");

            /* グリッドのX座標 */
            SetBindingProperty(PanelParam.GridX, ((dynamic)m_mainVM).GridX, "Value");

            /* グリッドのY座標 */
            SetBindingProperty(PanelParam.GridY, ((dynamic)m_mainVM).GridY, "Value");

            /* 線の幅 */
            SetBindingProperty(PanelParam.LineWidth, ((dynamic)m_mainVM).LineWidth, "Value");

            /* 線の透明度 */
            SetBindingProperty(PanelParam.LineAlpha, ((dynamic)m_mainVM).LineAlpha, "Value");

            /* 読み込み画像 */
            Binding bindingReadImage = new(nameof(PictureImage.Image), ((dynamic)m_mainVM).DrawImage, "Value");
            bindingReadImage.Parse += (object sender, ConvertEventArgs e) =>
            {
            };
            bindingReadImage.Format += (object sender, ConvertEventArgs e) =>
            {
                if (e.Value == null)
                {
                    e.Value = new Bitmap(100, 100);
                }
                else
                {
                    using MemoryStream stream = new((byte[])e.Value);
                    e.Value = new Bitmap(stream);
                }
            };
            PictureImage.DataBindings.Add(bindingReadImage);

            /* 線の色 */
            Binding bindingLineColor = new(nameof(ForeColor), ((dynamic)m_mainVM).LineColor, "Value");
            bindingLineColor.Parse += (object sender, ConvertEventArgs e) =>
            {
                Color color = (Color)e.Value;
                e.Value = $"{color.R:X02}{color.G:X02}{color.B:X02}{color.A:X02}";
            };
            bindingLineColor.Format += (object sender, ConvertEventArgs e) =>
            {
                e.Value = ColorTranslator.FromHtml($"#{((string)e.Value).Substring(0, 6)}");
            };
            PanelParam.LineColor.DataBindings.Add(bindingLineColor);

        }

        /// <summary>
        /// 指定したコントロールのプロパティに、データソースのプロパティをバインドします。
        /// この関数ではコントロールの種類によって、コントロールのプロパティ名を自動で取得します。
        /// </summary>
        /// <param name="control">コントロール</param>
        /// <param name="dataSource">データソース</param>
        /// <param name="dataMember">データソースのプロパティ名</param>
        /// <param name="formattingEnabled">表示データの書式を指定</param>
        private void SetBindingProperty(Control control, object dataSource, string dataMember, bool formattingEnabled = false)
        {

            /* コントロールのプロパティ名を取得 */
            string propertyName;
            if (control is NumericUpDown)
            {
                propertyName = nameof(NumericUpDown.Value);
            }
            else
            {
                propertyName = nameof(Control.Text);
            }

            /* バインド */
            SetBindingProperty(control, propertyName, dataSource, dataMember, formattingEnabled);

        }

        /// <summary>
        /// 指定したコントロールのプロパティに、データソースのプロパティをバインドします。
        /// </summary>
        /// <param name="control">コントロール</param>
        /// <param name="propertyName">コントロールのプロパティ名</param>
        /// <param name="dataSource">データソース</param>
        /// <param name="dataMember">データソースのプロパティ名</param>
        /// <param name="formattingEnabled">表示データの書式を指定</param>
        private void SetBindingProperty(Control control, string propertyName, object dataSource, string dataMember, bool formattingEnabled = false)
        {
            if (dataSource?.GetType().GetProperty(dataMember) != null)
            {
                control.DataBindings.Add(propertyName, dataSource, dataMember, formattingEnabled);
            }
        }

        #endregion

    }

}
